/* -*-c++-*- Copyright (C) 2018 Advanced Driver Information Technology.
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/


#include "WLPointer.hpp"
#include "WLSurfaceEvent.hpp"

using namespace WaylandBackend;

const struct wl_pointer_listener WLPointer::_pointerListener =
{
        WLPointer::pointerHandleEnter,
        WLPointer::pointerHandleLeave,
        WLPointer::pointerHandleMotion,
        WLPointer::pointerHandleButton,
        WLPointer::pointerHandleAxis
};

WLPointer::WLPointer(struct wl_seat *wlSeat, struct wl_event_queue *eventQ):
_wlPointer(NULL)
,_wlSeat(wlSeat)
,_pointerFocus(NULL)

{
    if (NULL != eventQ)
    {
        struct wl_seat* wrappedSeat = (struct wl_seat*)
                                wl_proxy_create_wrapper((void *)wlSeat);
        if (NULL != wrappedSeat)
        {
            wl_proxy_set_queue((struct wl_proxy*)wrappedSeat, eventQ);
            _wlPointer = (struct wl_pointer*) wl_seat_get_pointer(wrappedSeat);
            wl_proxy_wrapper_destroy(wrappedSeat);
        }
    }
    else
    {
        /*
         * It is not good idea to use default queue. default queue can be
         * dispatched by multiple threads
         */
        _wlPointer = (struct wl_pointer*) wl_seat_get_pointer(wlSeat);
    }
    if (NULL != _wlPointer)
        wl_pointer_add_listener(_wlPointer, &_pointerListener, (void*)this);
}

void
WLPointer::clearIfFocusMatches(struct wl_surface *wlSurface)
{
    if (_pointerFocus == wlSurface)
        _pointerFocus = NULL;
}

void
WLPointer::pointerHandleEnter(void* data, struct wl_pointer* wlPointer,
                              uint32_t serial, struct wl_surface* wlSurface,
                              wl_fixed_t sx, wl_fixed_t sy)
{
    WLPointer *pwlPointer = static_cast<WLPointer*>(data);
    WLSurfaceEvent *pInputlistener;
    if (NULL != wlSurface)
    {
        pInputlistener = static_cast<WLSurfaceEvent *>
                         (wl_surface_get_user_data(wlSurface));
        if (NULL != pInputlistener)
            pInputlistener->pointerEnter((void*)pwlPointer->_wlSeat,
                                         serial, wl_fixed_to_double(sx),
                                         wl_fixed_to_double(sy));
    }
    pwlPointer->_pointerFocus = wlSurface;
}

void
WLPointer::pointerHandleMotion(void* data, struct wl_pointer* wlPointer,
                               uint32_t time, wl_fixed_t sx, wl_fixed_t sy)
{
    WLPointer *pwlPointer = static_cast<WLPointer*>(data);
    WLSurfaceEvent *pInputlistener;
    if(NULL != pwlPointer->_pointerFocus)
    {
        pInputlistener = static_cast<WLSurfaceEvent *>
                         (wl_surface_get_user_data(pwlPointer->_pointerFocus));
        if (NULL != pInputlistener)
            pInputlistener->pointerMotion((void*)pwlPointer->_wlSeat, time,
                                           wl_fixed_to_double(sx),
                                           wl_fixed_to_double(sy));
    }
}

void
WLPointer::pointerHandleButton(void* data, struct wl_pointer* wlPointer,
                               uint32_t serial, uint32_t time, uint32_t button,
                               uint32_t state)
{
    WLPointer *pwlPointer = static_cast<WLPointer*>(data);
    WLSurfaceEvent *pInputlistener;
    if(NULL != pwlPointer->_pointerFocus)
    {
        pInputlistener = static_cast<WLSurfaceEvent *>
                         (wl_surface_get_user_data(pwlPointer->_pointerFocus));
        if (NULL != pInputlistener)
            pInputlistener->pointerButton((void*)pwlPointer->_wlSeat, serial,
                                          time, button, state);
    }
}

void
WLPointer::pointerHandleAxis(void* data, struct wl_pointer* wlPointer,
                             uint32_t time, uint32_t axis, wl_fixed_t value)
{
    WLPointer *pwlPointer = static_cast<WLPointer*>(data);
    WLSurfaceEvent *pInputlistener;
    if(NULL != pwlPointer->_pointerFocus)
    {
        pInputlistener = static_cast<WLSurfaceEvent *>
                       (wl_surface_get_user_data(pwlPointer->_pointerFocus));
        if (NULL != pInputlistener)
            pInputlistener->pointerAxis((void*)pwlPointer->_wlSeat, time,
                                        axis, value);
    }
}

void
WLPointer::pointerHandleLeave(void* data, struct wl_pointer* wlPointer,
                              uint32_t serial, struct wl_surface* wlSurface)
{
    WLPointer *pwlPointer = static_cast<WLPointer*>(data);
    WLSurfaceEvent *pInputlistener;
    if(NULL != pwlPointer->_pointerFocus)
    {
        pInputlistener = static_cast<WLSurfaceEvent *>
                       (wl_surface_get_user_data(pwlPointer->_pointerFocus));
        if (NULL != pInputlistener)
            pInputlistener->pointerLeave((void*)pwlPointer->_wlSeat, serial);
    }
}

WLPointer::~WLPointer()
{
    if (NULL != _wlPointer)
    {
        wl_pointer_destroy(_wlPointer);
    }
}
